#ifndef _DISTRIBUTOR_HPP
#define _DISTRIBUTOR_HPP 1

#include <chrono>
#include <vector>
#include <mutex>
#include <string>
#include <ctime>
#include <thread>
#include <atomic>
#include <unistd.h>

#include "JobTool.hpp"
#include "MyMonitor.hpp"
#include "Buffer.hpp"
#include "thrift/gen-cpp/Remote.h"
using namespace std;

#include <thrift/protocol/TBinaryProtocol.h>
#include <thrift/transport/TSocket.h>
#include <thrift/transport/TTransportUtils.h>
using namespace apache::thrift;
using namespace apache::thrift::protocol;
using namespace apache::thrift::transport;

class Distributor {

private:

	mutex _muResult;
	mutex _muTime;
	mutex _muJobsCount;
	Buffer * _buffer;

	Configurator * _conf;
	MyMonitor * _moni;
	Obj * _in;

	chrono::steady_clock::time_point _startPoint, _stopPoint;
	int _allWorker;
	long long int _duration;
	unsigned long long int _jobsCount;
	atomic<int> _runWorker;
	vector<string> _runningJobs;
	vector<int> _mem;
	
	//==================================================================
	struct WorkerConnect {
		RemoteClient * remote;
		RemoteClient * update;
		
		thread * proxy;

		string ip;
		int port;

		struct RemoteConf_s {
			boost::shared_ptr<TTransport> socket;
			boost::shared_ptr<TTransport> transport;
			boost::shared_ptr<TProtocol> protocol;
		} remoteConf;

		struct UpdateConf_s {
			boost::shared_ptr<TTransport> socket;
			boost::shared_ptr<TTransport> transport;
			boost::shared_ptr<TProtocol> protocol;
		} updateConf;
		
		void connect(string ip, int port) {
			remoteConf.socket = boost::shared_ptr<TTransport>(
				new TSocket(ip, port)
			);
			remoteConf.transport = boost::shared_ptr<TTransport>(
				new TBufferedTransport(remoteConf.socket)
			);
			remoteConf.protocol = boost::shared_ptr<TProtocol>(
				new TBinaryProtocol(remoteConf.transport)
			);
			remote = new RemoteClient(remoteConf.protocol);
			remoteConf.transport->open();

			updateConf.socket = boost::shared_ptr<TTransport>(
				new TSocket(ip, port)
			);
			updateConf.transport = boost::shared_ptr<TTransport>(
				new TBufferedTransport(updateConf.socket)
			);
			updateConf.protocol = boost::shared_ptr<TProtocol>(
				new TBinaryProtocol(updateConf.transport)
			);
			update = new RemoteClient(updateConf.protocol);
			updateConf.transport->open();
		}
	};
	
	vector<WorkerConnect> _connections;

	//==================================================================
	void monitor(bool endless = true) {
		while(true) {
			chrono::steady_clock::time_point stopP = chrono::steady_clock::now();
			_duration = chrono::duration_cast<chrono::seconds>(stopP - _startPoint).count();

			_moni->log(
				_runWorker.load(),
				_runningJobs,
				_allWorker,
				_duration,
				JobTool::toString(_result),
				*_conf,
				_buffer->size(),
				_buffer->getEnlargeCount(),
				_mem
			);

			if(endless == false) break;
			usleep(_moni->_intervall * 1000);
		}
	}

	//==================================================================
	void update(void) {
		while(true) {
			for (int id = 0; id < _allWorker; ++id) {
				usleep(_conf->syncBest * 1000);
				UpdateResult _return;
				
				_muResult.lock();
				_connections[id].update->update(_return, _result);
				
				if (
					(JobTool::compare(_result, _return.job) < 0 &&  _in->isMax) ||
					(JobTool::compare(_result, _return.job) > 0 && !_in->isMax)
				) {
					_result = _return.job;
				}
				_muResult.unlock();
				_mem[id] = _return.mem;
			}
		}
	}

	//==================================================================
	void worker(int id) {
		Job initJob = JobTool::create(_conf->jobTimeout);

		while (true) {
			Job job = _buffer->pop();
			if (JobTool::isEnd(job) == true) return;

			SolveResult _sreturn;
			
			_runningJobs[id] = JobTool::fixToString(job); // only for log and monitoring
			++_runWorker; // only for log and monitoring
			
			if (_conf->syncBest < 0) {
				_connections[id].remote->solve(_sreturn, initJob, job);
			} else {
				_connections[id].remote->solve(_sreturn, _result, job);
			}
			
			if (_sreturn.flag == SolveFlag::TIMEOUT) {
				_muJobsCount.lock();
				_buffer->push(_sreturn.newJobs);
				_jobsCount += _sreturn.newJobs.size();
				_muJobsCount.unlock();
			} else {
				_muResult.lock();
				
				if (
					(JobTool::compare(_result, _sreturn.newJobs[0]) < 0 &&  _in->isMax) ||
					(JobTool::compare(_result, _sreturn.newJobs[0]) > 0 && !_in->isMax)
				) {
					_result = _sreturn.newJobs[0];
				}
				_muResult.unlock();
			}
			_muJobsCount.lock();
			--_jobsCount;
			_muJobsCount.unlock();
			
			_runningJobs[id] = ""; // only for log and monitoring
			--_runWorker; // only for log and monitoring
		}
	}

	//==================================================================
	void farmer() {
		if(_conf->syncBest > 0) {
			thread updating = thread(&Distributor::update, this);
			updating.detach();
		}
		thread monitoring = thread(&Distributor::monitor, this, true);
		monitoring.detach();
		
		// zu anfang ist nur ein job dran ohne vorbelegung oder lazy Constraints
		_jobsCount = 1;
		_buffer->init();
		
		while( true ) {
			_muJobsCount.lock();
			if(_jobsCount == 0) {
				_muJobsCount.unlock();
				break;
			}
			_muJobsCount.unlock();
			usleep(1000000); // zum sparen von Resourcen
		}

		//end-Jobs
		_buffer->finish(_allWorker);
	}

public:

	Job _result;

	//==================================================================
	Distributor(Configurator * conf, MyMonitor * moni) {
		_conf = conf;
		_moni = moni;
		_allWorker = 0;
		_duration = 0;
		_result = JobTool::create(_conf->jobTimeout);
	}
	
	//==================================================================
	virtual ~Distributor() {
		for(int i=0; i<_allWorker; ++i) {
			_connections[i].remoteConf.transport->close();
			_connections[i].updateConf.transport->close();
			delete _connections[i].remote;
			delete _connections[i].update;
			delete _connections[i].proxy;
		}
		delete _buffer;
	}

	//==================================================================
	void push(string ip, int port) {
		WorkerConnect wo;
		wo.connect(ip, port);
		_connections.push_back(wo);
		_allWorker = _connections.size();
	}

	//==================================================================
	long long int run(Obj * in) {
		_in = in;
		_buffer = new Buffer(_in, _conf);
		_runWorker = 0;
		
		for (int i = 0; i < _allWorker; ++i) {
			_connections[i].remote->kill();
			_connections[i].remote->input(*_in, *_conf);
		}
		_runningJobs.resize(_allWorker);
		_mem.resize(_allWorker);

		_startPoint = chrono::steady_clock::now();
		
		for (int t = 0; t < _allWorker; ++t)
			 _connections[t].proxy = new thread(&Distributor::worker, this, t);

		thread tFarmer = thread(&Distributor::farmer, this);

		tFarmer.join();
		for (int t = 0; t < _allWorker; ++t)
			_connections[t].proxy->join();

		monitor(false);
		return _duration;
	}
	
	//==================================================================
	vector<long> getEnlargeCount() {
		return _buffer->getEnlargeCount();
	}
};
#endif
